## Using the BitPay PHP client

This SDK provides a convenient abstraction of BitPay's [cryptographically-secure API](https://bitpay.com/api) and allows payment gateway developers to focus on payment flow/e-commerce integration rather than on the specific details of client-server interaction using the API.  This SDK optionally provides the flexibility for developers to have control over important details, including the handling of private keys needed for client-server communication.

It also implements BitPay's remote client authentication and authorization strategy.  No private or shared-secret information is ever transmitted over the wire.

### Dependencies

You must have a BitPay merchant account to use this SDK.  It's free to [sign-up for a BitPay merchant account](https://bitpay.com/start).

If you need a test account, please visit https://test.bitpay.com/dashboard/signup and register for a BitPay merchant test account. Please fill in all questions, so you get a fully working test account.
If you are looking for a testnet bitcoin wallet to test with, please visit https://bitpay.com/wallet and
create a new wallet.
If you need testnet bitcoin please visit a testnet faucet, e.g. https://testnet.coinfaucet.eu/en/ or http://tpfaucet.appspot.com/

For more information about testing, please see https://bitpay.com/docs/testing

### Handling your client private key

Each client paired with the BitPay server requires a ECDSA key.  This key provides the security mechanism for all client interaction with the BitPay server. The public key is used to derive the specific client identity that is displayed on your BitPay dashboard.  The public key is also used for securely signing all API requests from the client.  See the [BitPay API](https://bitpay.com/api) for more information.

The private key should be stored in the client environment such that it cannot be compromised.  If your private key is compromised you should revoke the compromised client identity from the BitPay server and re-pair your client, see the [API tokens](https://bitpay.com/api-tokens) for more information.

To generate the configuration file required to load the SDK:

The [BitPay Config Generator](https://github.com/bitpay/php-bitpay-client-v2/blob/master/examples/ConfigGenerator.php) helps to generate the private key, as well as a environment file formatted in JSON or YML which contains all configuration requirements, that should be stored in the client local file system. It is not recommended to transmit the private key over any public or unsecure networks.

The comments in this script will assist you to create the environment file which you will be able to modify it later.

Once the Config Generator has run and generated the Json/Yml correctly, read the console output and follow the instructions in order to pair your new tokens.

The environment file can be either generated by the script mentioned avobe or created manually by copying the following Json or YML structure:

JSON:
```json
{
  "BitPayConfiguration": {
    "Environment": "",
    "EnvConfig": {
      "Test": {
        "PrivateKeyPath": "",
        "PrivateKeySecret": "",
        "ApiTokens": {
          "merchant": "",
          "payroll": ""
        }
      },
      "Prod": {
        "PrivateKeyPath": "",
        "PrivateKeySecret": "",
        "ApiTokens": {
          "merchant": "",
          "payroll": ""
        }
      }
    }
  }
}
```
##
YML:
```yml
BitPayConfiguration:
  Environment: null
  EnvConfig:
    Test:
      PrivateKeyPath: null
      PrivateKeySecret: null
      ApiTokens:
        merchant: null
        payroll: null
    Prod:
      PrivateKeyPath: null
      ApiTokens:
        merchant: null
        payroll: null
```

# Installation

## Composer

### Install Composer

```bash
curl -sS https://getcomposer.org/installer | php
```

### Install via composer by hand

Add to your composer.json file by hand.

```javascript
{
    ...
    "require": {
        ...
        "bitpay/sdk": "^5.0"
    }
    ...
}
```

Once you have added this, just run:

```bash
php composer.phar update bitpay/sdk
```

### Install using composer

```bash
php composer.phar require bitpay/sdk:^5.0
```

### Initializing your BitPay client

Once you have the environment file (JSON or YML previously generated) you can initialize the client on two different ways:

```php
// Provide the full path to the env file which you have previously stored securely.

$bitpay = BitPaySDK\Client::create()->withFile([FULL_PATH_TO_THE_CONFIG_FILE]);
```

```php
// Initialize with separate variables 
// and Private Key stored in file.

$bitpay = BitPaySDK\Client::create()->withData(
    BitPaySDK\Env.Test,
    "[FULL_PATH_TO_THE_PRIVATE_KEY]",
    new BitPaySDK\Tokens(
        "7UeQtMcsHamehE4gDZojUQbNRbSuSdggbH17sawtobGJ", //merchant
        "5j48K7pUrX5k59DLhRVYkCupgw2CtoEt8DBFrHo2vW47" //payroll
    ),
    "YourMasterPassword" //used to decrypt your private key, if encrypted
);
```
```php
// Initialize with separate variables 
// and Private Key as HEX string.

$bitpay = BitPaySDK\Client::create()->withData(
    BitPaySDK\Env.Test,
    "[PRIVATE_KEY_AS_HEX_STRING]",
    new BitPaySDK\Tokens(
        "7UeQtMcsHamehE4gDZojUQbNRbSuSdggbH17sawtobGJ", //merchant
        "5j48K7pUrX5k59DLhRVYkCupgw2CtoEt8DBFrHo2vW47" //payroll
    )
);
```
##
### Create an invoice

```php
$invoice = $bitpay->createInvoice(new Invoice(50.0, "USD"));

$invoiceUrl = $invoice->getURL();

$status = $invoice->getStatus();
```

> **WARNING**: 
If you get the following error when initiating the client for first time:
"500 Internal Server Error` response: {"error":"Account not setup completely yet."}"
Please, go back to your BitPay account and complete the required steps.
More info [here](https://support.bitpay.com/hc/en-us/articles/203010446-How-do-I-apply-for-a-merchant-account-)

### Retrieve an invoice

```php
$invoice = $bitpay->getInvoice($invoice->getId());
```

### Get exchange Rates

You can retrieve BitPay's [BBB exchange rates](https://bitpay.com/exchange-rates).

```php
$rates = $bitpay->getRates();

$rate = $rates->getRate(Currency::USD); //Always use the included Currency model to avoid typos

$rates->update();
```

See also the test package for more examples of API calls.

